<?php
/**
 * Product Attribute Guide Addon
 *
 * @author     D-THEMES
 * @package    WP Alpha Core Framework
 * @subpackage Core
 * @since      1.2.0
 */

defined( 'ABSPATH' ) || die;

if ( ! class_exists( 'Alpha_Product_Attribute_Guide' ) ) {
	class Alpha_Product_Attribute_Guide extends Alpha_Base {

		/**
		 * Constructor
		 *
		 * @since 1.2.0
		 */
		public function __construct() {

			// Add theme options
			add_filter( 'alpha_customize_fields', array( $this, 'add_customize_fields' ) );
			add_filter(
				'alpha_customize_sections',
				function( $sections ) {
					$sections['attr_guide'] = array(
						'title'    => esc_html__( 'Attribute Guide', 'alpha-core' ),
						'panel'    => 'features',
						'priority' => 18,
					);
					return $sections;
				}
			);
			if ( function_exists( 'alpha_set_default_option' ) ) {
				alpha_set_default_option( 'guide_display_type', 'tab' );
			}

			// Attribute guide
			if ( is_admin() && 'edit.php' == $GLOBALS['pagenow'] && isset( $_REQUEST['post_type'] ) && 'product' == $_REQUEST['post_type'] && isset( $_REQUEST['page'] ) && 'product_attributes' == $_REQUEST['page'] ) {
				add_action( 'woocommerce_after_add_attribute_fields', array( $this, 'add_guide_options' ) );
				add_action( 'woocommerce_after_edit_attribute_fields', array( $this, 'edit_guide_options' ) );

				$this->update_guide_options();
			}

			if ( ! function_exists( 'alpha_get_option' ) || 'tab' == alpha_get_option( 'guide_display_type' ) ) {
				add_filter( 'woocommerce_product_tabs', array( $this, 'attribute_guide' ), 100 );
			}

			if ( function_exists( 'alpha_get_option' ) && 'lightbox' == alpha_get_option( 'guide_display_type' ) && defined( 'ELEMENTOR_VERSION' ) ) {
				// Clear popup cache after template save
				if ( wp_doing_ajax() ) {
					add_action( 'elementor/document/before_save', array( $this, 'clear_popup_cache' ), 20, 2 );
				}
			}
		}

		/**
		 * Add fields for compare
		 *
		 * @param {Array} $fields
		 *
		 * @param {Array} $fields
		 *
		 * @since 1.0
		 */
		public function add_customize_fields( $fields ) {
			$fields['cs_attr_guide_about_title'] = array(
				'section' => 'attr_guide',
				'type'    => 'custom',
				'label'   => '',
				'default' => '<h3 class="options-custom-title option-feature-title">' . esc_html__( 'About This Feature', 'alpha-core' ) . '<span class="doc-trigger"><a href="https://d-themes.com/wordpress/udesign/documentation/2022/02/13/product-attribute-guide/" target="_blank" aria-label="' . esc_attr__( 'Documentation', 'alpha-core' ) . '"><span class="' . ALPHA_ICON_PREFIX . '-icon-file-alt"></span></a></span>' . '</h3>',
			);
			$fields['cs_attr_guide_desc']        = array(
				'section' => 'attr_guide',
				'type'    => 'custom',
				'label'   => sprintf( esc_html__( 'The attribute guide feature was developed to add a product attribute description such as %1$ssize chart%2$s to a product page.', 'alpha-core' ), '<strong>', '</strong>' ),
				'default' => '<p class="options-custom-description option-feature-description"><img class="description-image" src="' . ALPHA_ASSETS . '/images/admin/customizer/guide.jpg' . '" alt="' . esc_html__( 'Theme Option Descrpition Image', 'alpha-core' ) . '"></p>',
			);
			$fields['cs_attr_guide_advanced']    = array(
				'section' => 'attr_guide',
				'type'    => 'custom',
				'label'   => '',
				'default' => '<h3 class="options-custom-title">' . esc_html__( 'Attribute Guide', 'alpha-core' ) . '</h3>',
			);
			$fields['guide_display_type']        = array(
				'section'   => 'attr_guide',
				'type'      => 'radio_buttonset',
				'label'     => esc_html__( 'Guide Layout', 'alpha-core' ),
				'tooltip'   => esc_html__( 'Select attribute guide layout. Add an additional tab in product tabs or open a lightbox at guide button clicks.', 'alpha-core' ),
				'transport' => 'refresh',
				'choices'   => array(
					'tab'      => esc_html__( 'Additional Tab', 'alpha-core' ),
					'lightbox' => esc_html__( 'Lightbox', 'alpha-core' ),
				),
			);
			return $fields;
		}

		/**
		 * The wc product attribute to add guide otpions.
		 *
		 * @since 1.2.0
		 */
		public function add_guide_options() {
			// Get blocks
			$posts = get_posts(
				array(
					'post_type'   => ALPHA_NAME . '_template',
					'meta_key'    => ALPHA_NAME . '_template_type',
					'meta_value'  => 'block',
					'numberposts' => -1,
				)
			);
			sort( $posts );
			?>
			<div class="form-field">
				<label for="guide_text"><?php esc_html_e( 'Guide Button Label', 'alpha-core' ); ?></label>
				<textarea name="guide_text" id="guide_text"></textarea>
				<?php if ( function_exists( 'alpha_get_icons_store_url' ) ) : ?>
				<p class="description"><?php echo sprintf( esc_html__( 'Link text for showing guide content. You can also input icon html from %1$shere%2$s.', 'alpha-core' ), '<a href="' . esc_url( alpha_get_icons_store_url() ) . '" target="_blank">', '</a>' ); ?></p>
				<?php endif; ?>
			</div>
			<div class="form-field">
				<label for="guide_block"><?php esc_html_e( 'Guide Content', 'alpha-core' ); ?></label>
				<select name="guide_block" id="guide_block">
					<option value=""></option>
				<?php foreach ( $posts as $post ) : ?>
					<option value="<?php echo esc_attr( $post->ID ); ?>"><?php echo esc_html( $post->post_title ); ?></option>
				<?php endforeach; ?>
				</select>
				<?php
				if ( function_exists( 'alpha_get_block_edit_links' ) ) {
					echo alpha_get_block_edit_links( '.form-field' );
				}
				?>
				<p class="description"><?php echo sprintf( esc_html__( 'It supports 2 display types: product additional tab or a new lightbox in %1$stheme option%2$s.', 'alpha-core' ), '<a href="' . esc_url( admin_url( 'customize.php?#attr_guide' ) ) . '" target="_blank">', '</a>' ); ?></p>
			</div>
			<?php
		}

		/**
		 * The wc product attribute edit guide options.
		 *
		 * @since 1.2.0
		 */
		public function edit_guide_options() {
			$guide_block = isset( $_POST['guide_block'] ) ? absint( $_POST['guide_block'] ) : ''; // WPCS: input var ok, CSRF ok.
			$guide_text  = isset( $_POST['guide_text'] ) ? alpha_strip_script_tags( $_POST['guide_text'] ) : ''; // WPCS: input var ok, CSRF ok.
			$edit        = isset( $_GET['edit'] ) ? absint( $_GET['edit'] ) : 0;

			if ( $edit ) {
				global $wpdb;
				$attribute = $wpdb->get_row(
					$wpdb->prepare( "SELECT attribute_name, attribute_label FROM {$wpdb->prefix}woocommerce_attribute_taxonomies WHERE attribute_id = %d", $edit )
				);

				if ( $attribute ) {
					$att_name = $attribute->attribute_name;

					$alpha_pa_blocks = get_option( 'alpha_pa_blocks', array() );
					if ( isset( $alpha_pa_blocks[ $att_name ] ) ) {
						$guide_block = $alpha_pa_blocks[ $att_name ]['block'];
						$guide_text  = $alpha_pa_blocks[ $att_name ]['text'];
					}
				}
			}

			// Get blocks
			$posts = get_posts(
				array(
					'post_type'   => ALPHA_NAME . '_template',
					'meta_key'    => ALPHA_NAME . '_template_type',
					'meta_value'  => 'block',
					'numberposts' => -1,
				)
			);
			sort( $posts );

			// Form
			?>
			<tr class="form-field">
				<th scope="row" valign="top">
					<label for="guide_text"><?php esc_html_e( 'Guide Button Label', 'alpha-core' ); ?></label>
				</th>
				<td>
					<textarea name="guide_text" id="guide_text" placeholder="<?php echo esc_html( '<i class="' . THEME_ICON_PREFIX . '-icon-ruler2"></i>' ) . esc_attr( $attribute->attribute_label ) . esc_html__( ' Guide', 'alpha-core' ); ?>"><?php echo wp_unslash( alpha_strip_script_tags( $guide_text ) ); ?></textarea>
					<?php if ( function_exists( 'alpha_get_icons_store_url' ) ) : ?>
					<p class="description"><?php echo sprintf( esc_html__( 'Link text for showing guide content. You can also input icon html from %1$shere%2$s.', 'alpha-core' ), '<a href="' . esc_url( alpha_get_icons_store_url() ) . '" target="_blank">', '</a>' ); ?></p>
					<?php endif; ?>
				</td>
			</tr>
			<tr class="form-field">
				<th scope="row" valign="top">
					<label for="guide_block"><?php esc_html_e( 'Guide Content', 'alpha-core' ); ?></label>
				</th>
				<td>
					<select name="guide_block" id="guide_block">
						<option value=""></option>
						<?php foreach ( $posts as $post ) : ?>
							<option value="<?php echo esc_attr( $post->ID ); ?>" <?php selected( $guide_block, $post->ID ); ?>><?php echo esc_html( $post->post_title ); ?></option>
						<?php endforeach; ?>
					</select>
					<?php
					if ( function_exists( 'alpha_get_block_edit_links' ) ) {
						echo alpha_get_block_edit_links( '.form-field' );
					}
					?>
					<p class="description"><?php echo sprintf( esc_html__( 'It supports 2 display types: product additional tab or a new lightbox in %1$stheme option%2$s.', 'alpha-core' ), '<a href="' . esc_url( admin_url( 'customize.php?#attr_guide' ) ) . '" target="_blank">', '</a>' ); ?></p>
				</td>
			</tr>
			<?php
		}

		/**
		 * Add attribute guide
		 *
		 * @since 1.2.0
		 */
		public function attribute_guide( $tabs ) {

			// Guide block
			global $product;
			$attributes      = $product->get_attributes();
			$alpha_pa_blocks = get_option( 'alpha_pa_blocks' );

			foreach ( $attributes as $key => $attribute ) {
				$name = substr( $key, 3 );
				if ( isset( $alpha_pa_blocks[ $name ] ) &&
					isset( $alpha_pa_blocks[ $name ]['block'] ) && $alpha_pa_blocks[ $name ]['block'] && 'publish' == get_post_status( $alpha_pa_blocks[ $name ]['block'] ) ) {

					$button_text = ! empty( $alpha_pa_blocks[ $name ]['text'] ) ? $alpha_pa_blocks[ $name ]['text'] : ( '<i class="' . THEME_ICON_PREFIX . '-icon-ruler2"></i>' . wc_attribute_label( $key ) . esc_html__( ' Guide', 'alpha-core' ) );

					$tabs[ 'alpha_pa_block_' . $name ] = apply_filters(
						"alpha_product_attribute_{$name}_guide",
						array(
							'title'    => alpha_strip_script_tags( $button_text ),
							'priority' => 28,
							'callback' => 'alpha_wc_product_custom_tab',
							'block_id' => absint( $alpha_pa_blocks[ $name ]['block'] ),
						)
					);
				}
			}
			return $tabs;
		}

		/**
		 * The product attribute to add guide options
		 *
		 * @since 1.2.0
		 */
		public function update_guide_options() {
			//  Add, edit, or delete guide options
			$guide_block = ! empty( $_POST['guide_block'] ) ? absint( $_POST['guide_block'] ) : ''; // WPCS: input var ok, CSRF ok.
			$guide_text  = ! empty( $_POST['guide_text'] ) ? alpha_strip_script_tags( $_POST['guide_text'] ) : ''; // WPCS: input var ok, CSRF ok.
			$att_name    = ! empty( $_POST['attribute_name'] ) ? wc_sanitize_taxonomy_name( wp_unslash( $_POST['attribute_name'] ) ) : ( ! empty( $_POST['attribute_label'] ) ? wc_sanitize_taxonomy_name( $_POST['attribute_label'] ) : '' ); // WPCS: input var ok, CSRF ok, sanitization ok.

			$alpha_pa_blocks = get_option( 'alpha_pa_blocks', array() );
			if ( ! empty( $_POST['add_new_attribute'] ) || ( ! empty( $_POST['save_attribute'] ) && ! empty( $_GET['edit'] ) ) ) { // WPCS: CSRF ok.
				$alpha_pa_blocks[ $att_name ] = array(
					'block' => $guide_block,
					'text'  => $guide_text,
				);
			} elseif ( ! empty( $_GET['delete'] ) && isset( $alpha_pa_blocks[ $att_name ] ) ) {
				unset( $alpha_pa_blocks[ $att_name ] );
			}
			update_option( 'alpha_pa_blocks', $alpha_pa_blocks );
		}

		public function clear_popup_cache( $self, $data ) {
			if ( empty( $_REQUEST['editor_post_id'] ) ) {
				return;
			}
			$post_id = absint( $_REQUEST['editor_post_id'] );

			$alpha_pa_blocks = get_option( 'alpha_pa_blocks' );
			$update_require  = false;

			foreach ( $alpha_pa_blocks as $attr => $values ) {
				if ( ! empty( $values['block'] ) && $post_id == $values['block'] ) {
					$update_require = true;
				}
			}

			if ( $update_require ) {
				set_theme_mod( 'popup_last_time', time() + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
			}
		}
	}
}

Alpha_Product_Attribute_Guide::get_instance();
